/*==========================================================================
  MEBLIT.C

  Example code to perform a monochrome expansion bitblt. A monochrome line
  of data (displayed at near the bottom-left region of the screen) acts as
  the source data to be expanded. The resultant expansion is shown near the
  top-left region of the screen.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"

#define INSTANCE 0

/* --------------------------------------------------------------------------
  reversedata - reverse the bits of the given data

  This is a general utility.
-------------------------------------------------------------------------- */
unsigned long reversedata (unsigned long data, int nbits)
{
    unsigned long newdata, bit, mask, shifter;
    int i;

    if (nbits > 32) nbits = 32;

    newdata = 0;
    shifter = 0;
    mask = 1;
    for (i = 0; i < nbits; i++)
    {
        bit = (data & mask) >> shifter;
        newdata = (newdata << 1) | bit;

        shifter++;
        mask = mask << 1;
    }

    return (newdata);
}

/* --------------------------------------------------------------------------
  convertpattern - rearrange fill pattern for 4 bpp modes

  Since the default CRTC pixel order is MSB to LSB in each byte for 4 bpp
  modes, fill patterns need converting.
-------------------------------------------------------------------------- */
unsigned long convertpattern (unsigned long pattern)
{
    unsigned long byte1, byte2, byte3, byte4;

    // don't convert if not 4 bpp mode
    if (modeinfo.bpp != 4)
    {
        return (pattern);
    }

    byte1 = reversedata (pattern & 0xFF, 8);
    byte2 = reversedata ((pattern & 0xFF00) >> 8, 8) << 8;
    byte3 = reversedata ((pattern & 0xFF0000) >> 16, 8) << 16;
    byte4 = reversedata ((pattern & 0xFF000000) >> 24, 8) << 24;

    return (byte4 | byte3 | byte2 | byte1);
}


/* Main C program */

int main (int argc, char *argv[])
{
    int srcx, srcy, srcwidth, srcheight;
    int dstx, dsty, dstwidth, dstheight;
    int loop, hostwrts;
    unsigned long color_depth;
    unsigned long offset;
    unsigned long bitmap[13];

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // Process the command line arguments to override default resolution
    // and color depth settings.
    process_command_line (argc, argv);

    // set an accelerator mode
    if (open_mode (gmode_res, PITCH_XRES, gclr_depth) != NO_ERROR)
    {
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // Check for 24 bpp mode - Monochrome expansions are not directly
    // supported in 24 bpp modes
    if (modeinfo.bpp == 24)
    {
        // disable accelerator mode and switch back to VGA text mode
        close_mode ();

        printf ("Monochrome expansions are not directly supported in 24 bpp "
                "modes.\n");
        return (1);
    }

    // initialize standard engine context
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // Fill the bitmap to be used as monochrome source data - the letter 'a'
    bitmap[0] = 0x0003FFFF;
    bitmap[1] = 0x000C0006;
    bitmap[2] = 0xF0300018;
    bitmap[3] = 0x38C3F860;
    bitmap[4] = 0x3318398E;
    bitmap[5] = 0x0C780630;
    bitmap[6] = 0x319F18FC;
    bitmap[7] = 0xC60F630F;
    bitmap[8] = 0x3E0D8C06;
    bitmap[9] = 0xC7E6FFFB;
    bitmap[10] = 0xFFF8000D;
    bitmap[11] = 0x0000001F;
    bitmap[12] = 0x00000000;

    // Setup source and destination coordinates and sizes:
    //
    //    Note that these coordinates are in the modal bpp. An odd sized
    //    bitmap has been chosen to show how to handle remainder pixels (a
    //    total bit size that doesn't divide evenly into 32). The bitmap
    //    must be loaded into video memory via the HOST register. Each
    //    write transfers 32 pixels.
    //
    dstx = 10;          // resultant coordinate for blit expansion
    dsty = 20;
    dstwidth = 17;      // size of monochrome bitmap
    dstheight = 21;

    srcx = 0;           // location of monochrome blit source data (bitmap)
    srcy = 400;
    srcwidth = dstwidth * dstheight; // linear size of monochrome source data
    srcheight = 1;

    // Determine qword offset of coordinate (srcx, srcy) - this is the format
    // required for the DST_OFF_PITCH and SRC_OFF_PITCH registers.
    //
    offset = get_xy_offset (srcx, srcy) / 8;

    // Determine number of host writes needed. Each host write produces 32
    // pixels (in monochrome). If the number of host writes is not even,
    // the remainder in the last host write will be thrown away by the engine
    hostwrts = srcwidth / 32;
    if (srcwidth != (hostwrts * 32))
    {
        // round up to handle pixel remainder
        hostwrts++;
    }

    /* ---- Setup a monochrome source pattern ---- */

    // This is done by drawing host data linearly in memory. The pattern will
    // be placed at (srcx, srcy) so that it is visible on the screen. The
    // linear source data will be monochrome expanded into a DSTWIDTH x
    // DSTHEIGHT rectangle (dstwidth * dstheight bits). A "1" bit in the
    // source data will be expanded to the foreground color and a "0" will be
    // expanded to the background color. The resultant pattern will be an 'a'
    // with a border.

    // set engine host, source, and destination to 1 bpp; also set the
    // byte order bit which affects 1 and 4 bpp modes
    wait_for_fifo (2);
    if (modeinfo.bpp == 4)
    {
        // For 4 bpp modes, the CRTC byte format is reversed
        regw (DP_PIX_WIDTH, BYTE_ORDER_MSB_TO_LSB |
                            HOST_1BPP | SRC_1BPP | DST_1BPP);
    }
    else
    {
        regw (DP_PIX_WIDTH, BYTE_ORDER_LSB_TO_MSB |
                            HOST_1BPP | SRC_1BPP | DST_1BPP);
    }

    // set engine to use monochrome host registers for source data:
    //
    //      - for monochrome host data of "1", the foreground source channel
    //        is used
    //      - for monochrome host data of "0", the background source channel
    //        is used
    //
    regw (DP_SRC, MONO_SRC_HOST | FRGD_SRC_FRGD_CLR | BKGD_SRC_BKGD_CLR);

    // set foreground mix to ONE and background mix to ZERO
    set_fg_mix (ONE_MIX);
    set_bg_mix (ZERO_MIX);

    // setup monochrome host data to be copied to screen memory at
    // (srcx, srcy):
    //
    //      To avoid converting 8 bpp coordinates into 1 bpp coordinates and
    //      expanding the engine scissors, the 'offset' value calculated
    //      above will allow drawing at (srcx, srcy) by setting the
    //      coordinate registers DST_X, DST_Y to (0, 0)
    //

    // add offset without altering the pitch
    wait_for_fifo (5);
    regw (DST_OFF_PITCH, (regr (DST_OFF_PITCH) & 0xFFC00000) | offset);

    regw (DST_X, 0);
    regw (DST_Y, 0);
    regw (DST_HEIGHT, 1);
    regw (DST_WIDTH, srcwidth);

    // copy host data to screen memory - pattern order is LSB to MSB
    for (loop = 0; loop < hostwrts; loop++)
    {
        wait_for_fifo (1);
        regw (HOST_DATA0, convertpattern (bitmap[loop]));
    }

    // insure host transfer is done
    wait_for_idle ();

    // restore destination offset to zero
    regw (DST_OFF_PITCH, regr (DST_OFF_PITCH) & 0xFFC00000);


    /* ---- Perform blit expansion from the monochrome source data ---- */

    // determine modal color depth for destination
    switch (modeinfo.bpp)
    {
        case 4:
            color_depth = DST_4BPP;
            break;
        case 8:
            color_depth = DST_8BPP;
            break;
        case 16:
            if (modeinfo.depth == 555)
            {
                color_depth = DST_15BPP;
            }
            else
            {
                color_depth = DST_16BPP;
            }
            break;
        case 32:
            color_depth = DST_32BPP;
            break;
    }

    // set engine source color depth to 1 bpp and destination color depth
    // to the modal color depth
    if (modeinfo.bpp == 4)
    {
        // For 4 bpp modes, the CRTC byte format is reversed
        regw (DP_PIX_WIDTH, BYTE_ORDER_MSB_TO_LSB |
                            HOST_1BPP | SRC_1BPP | color_depth);
    }
    else
    {
        regw (DP_PIX_WIDTH, BYTE_ORDER_LSB_TO_MSB |
                            HOST_1BPP | SRC_1BPP | color_depth);
    }

    // set engine to use blit registers for monochrome source data:
    //
    //      - for monochrome blit of "1", the foreground color register
    //        is used
    //      - for monochrome blit of "0", the background color register
    //        is used
    //
    regw (DP_SRC, MONO_SRC_BLIT | FRGD_SRC_FRGD_CLR | BKGD_SRC_BKGD_CLR);

    // set foreground and background mix to OVERPAINT
    set_fg_mix (S_MIX);
    set_bg_mix (S_MIX);

    // foreground color = WHITE
    set_fg_color (get_color_code (WHITE));

    // background color = LIGHTBLUE
    set_bg_color (get_color_code (LIGHTBLUE));

    // add offset without altering the pitch
    wait_for_fifo (11);
    regw (SRC_OFF_PITCH, (regr (SRC_OFF_PITCH) & 0xFFC00000) | offset);

    // set source attribute to linear source data format
    regw (SRC_CNTL, SRC_LINEAR_ENABLE);

    // set source registers to monochrome data location
    regw (SRC_X, 0);
    regw (SRC_Y, 0);
    regw (SRC_HEIGHT1, 1);
    regw (SRC_WIDTH1, srcwidth);

    // set destination draw direction and enable engine tiling:
    regw (DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT |
                    DST_X_TILE | DST_Y_TILE);

    // set destination start coordinates and rectangle size to fill
    regw (DST_X, dstx);
    regw (DST_Y, dsty);
    regw (DST_HEIGHT, dstheight);

    // since the X & Y tiling is enabled in DST_CNTL, DST_X and DST_Y will
    // be automatically updated by the engine after each operation
    for (loop = 0; loop < 10; loop++)
    {
        wait_for_fifo (1);
        regw (DST_WIDTH, dstwidth);   // do monochrome expansion blit
    }

    // restore source offset to zero
    wait_for_fifo (1);
    regw (SRC_OFF_PITCH, regr (SRC_OFF_PITCH) & 0xFFC00000);

    // wait for a carriage return
    getch ();

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

