/*==========================================================================
  POLYGON.C

  Example code to draw a filled polygon.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"

#define INSTANCE 0

int main (int argc, char *argv[])
{
    unsigned long offset, color_depth, mono_pitch;
    unsigned long save_dst_off_pitch, save_src_off_pitch;
    int width, height;
    int box_x, box_y;
    int x1, y1, x2, y2, xtemp, ytemp;
    int i;
    POINT points[6];

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // Process the command line arguments to override default resolution
    // and color depth settings.
    process_command_line (argc, argv);

    // set an accelerator mode
    if (open_mode (gmode_res, PITCH_XRES, gclr_depth) != NO_ERROR)
    {
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // Check for 24 bpp mode - Polygons are not supported in 24 bpp modes
    if (modeinfo.bpp == 24)
    {
        // disable accelerator mode and switch back to VGA text mode
        close_mode ();

        printf("Polygons are not supported in 24 bpp modes.\n");
        return (1);
    }

    // setup engine context and clear screen
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // Polygon shape is a 5 sided pentagon - resolution independent

    points[0].x = modeinfo.xres / 2;
    points[0].y = modeinfo.yres / 12;
    points[1].x = (modeinfo.xres / 2) + (modeinfo.xres / 4);
    points[1].y = modeinfo.yres / 2;
    points[2].x = (modeinfo.xres / 2) + (modeinfo.xres / 8);
    points[2].y = modeinfo.yres - (modeinfo.yres / 12);
    points[3].x = (modeinfo.xres / 2) - (modeinfo.xres / 8);
    points[3].y = modeinfo.yres - (modeinfo.yres / 12);
    points[4].x = (modeinfo.xres / 2) - (modeinfo.xres / 4);
    points[4].y = modeinfo.yres / 2;
    points[5].x = points[0].x;
    points[5].y = points[0].y;

    // determine width, height, x & y offsets
    width = points[1].x - points[4].x + 1;
    height = points[2].y - points[0].y + 1;
    box_x = points[4].x;
    box_y = points[0].y;


    /* ---- Draw filled polygon ---- */

    // Save the default settings for the source and destination trajectory
    // offset and pitch registers.
    wait_for_idle ();
    save_dst_off_pitch = regr (DST_OFF_PITCH);
    save_src_off_pitch = regr (SRC_OFF_PITCH);

    // Calculate qword offset address of the start of off-screen memory
    offset = get_xy_offset (0, modeinfo.yres) / 8;

    // Calculate the monochrome pitch for the off_screen region where the
    // 1bpp polygon boundry lines will be drawn. Insure that this pitch
    // is an integer multiple of 64 (i.e. round up to the nearest
    // multiple of 64).
    mono_pitch = (unsigned long) ((((modeinfo.xres) + 63) / 64) * 64);
    mono_pitch = ((mono_pitch / 8) << 22) & 0xFFC00000;

    // 1. Set destination to 1 bpp for memory clearing and outline drawing
    wait_for_fifo (16);
    regw (DP_PIX_WIDTH, DST_1BPP);


    // 2. Set scissors to outline drawing region
    regw (SC_LEFT, 0);
    regw (SC_TOP, 0);
    regw (SC_RIGHT, width - 1);
    regw (SC_BOTTOM, height - 1);


    // 3. Clear off-screen memory where polygon outlines are to be drawn

    // insure engine idleness for GUI register reading
    wait_for_idle ();

    // set destination operations to off-screen memory
    regw (DST_OFF_PITCH, mono_pitch | offset);

    // clear memory
    regw (DP_SRC, FRGD_SRC_FRGD_CLR);
    regw (DP_MIX, FRGD_MIX_ZERO | BKGD_MIX_ZERO);
    regw (DST_X, 0);
    regw (DST_Y, 0);
    regw (DST_HEIGHT, height);
    regw (DST_WIDTH, width);


    // 4. Set mix to XOR
    regw (DP_MIX, FRGD_MIX_D_XOR_S | BKGD_MIX_ZERO);
    regw (DP_FRGD_CLR, 1);


    // 5. Set the DST_POLYGON_ENABLE bit, clear LAST_PEL bit
    regw (DST_CNTL, DST_POLYGON_ENABLE |
                    DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);


    // 6. Draw all polygon outlines from top to bottom with LAST_PEL_OFF
    for (i = 0; i < 5; i++)
    {
        x1 = points[i].x - box_x;
        y1 = points[i].y - box_y;
        x2 = points[i+1].x - box_x;
        y2 = points[i+1].y - box_y;

        // swap points if direction is not top to bottom
        if (y1 > y2)
        {
            ytemp = y1;
            y1 = y2;
            y2 = ytemp;

            xtemp = x1;
            x1 = x2;
            x2 = xtemp;
        }

        draw_line (x1, y1, x2, y2);
    }

    // 7. Set scissors to the final destination area
    wait_for_fifo (4);
    regw (SC_LEFT, 0);
    regw (SC_TOP, 0);
    regw (SC_RIGHT, modeinfo.xres - 1);
    regw (SC_BOTTOM, modeinfo.yres - 1);


    // 8. Set all necessary destination context registers

    // insure engine idleness for GUI register reading
    wait_for_idle ();

    // Set source operations to off-screen memory: Set source trajectory
    // pitch to that of the off-screen polygon boundary region, the offset
    // to the start of this off-screen memory region.
    regw (SRC_OFF_PITCH, mono_pitch | offset);

    // Set destination operations to on-screen memory: Set the destination
    // trajectory pitch to the native pitch, the offset to the start of
    // the display in on-screen memory.
    regw (DST_OFF_PITCH, save_dst_off_pitch & 0xFFC00000);

    // set destination color depth to current mode
    switch (modeinfo.bpp)
    {
        case 4:
            color_depth = DST_4BPP;
            break;
        case 8:
            color_depth = DST_8BPP;
            break;
        case 16:
            if (modeinfo.depth == DEPTH_SUPPORT_555)    // 555 color weighting
            {
                color_depth = DST_15BPP;
            }
            else                          // 565 color weighting
            {
                color_depth = DST_16BPP;
            }
            break;
        case 32:
            color_depth = DST_32BPP;
            break;
    }
    regw (DP_PIX_WIDTH, SRC_1BPP | color_depth);

    // set desired mix and color values
    set_fg_mix (S_MIX);
    set_bg_mix (D_MIX);
    set_fg_color (get_color_code (LIGHTRED));


    // 9. Setup blit source registers to point to polygon outline area
    wait_for_fifo (8);
    regw (SRC_X, 0);
    regw (SRC_Y, 0);
    regw (SRC_HEIGHT1, height);
    regw (SRC_WIDTH1, width);

    // 10. Setup blit destination registers to point to final destination area
    regw (DST_X, box_x);
    regw (DST_Y, box_y);
    regw (DST_HEIGHT, height);

    // 11. Blit
    regw (DST_WIDTH, width);

    // Restore DST_OFF_PITCH and SRC_OFF_PITCH before drawing the polygon
    // outline.
    wait_for_fifo (2);
    regw (DST_OFF_PITCH, save_dst_off_pitch);
    regw (SRC_OFF_PITCH, save_src_off_pitch);


    // Draw a WHITE border around shape using regular lines
    //
    //    This polygon bit in DST_CNTL must be turned off for regular line
    //    drawing.
    //
    wait_for_fifo (1);
    regw (DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    set_fg_color (get_color_code (WHITE));
    for (i = 0; i < 5; i++)
    {
        x1 = points[i].x;
        y1 = points[i].y;
        x2 = points[i+1].x;
        y2 = points[i+1].y;

        // swap points if direction is not top to bottom
        if (y1 > y2)
        {
            ytemp = y1;
            y1 = y2;
            y2 = ytemp;

            xtemp = x1;
            x1 = x2;
            x2 = xtemp;
        }

        draw_line (x1, y1, x2, y2);
    }

    // wait for a carriage return
    getch ();

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

