/*==========================================================================
  CHAIN.C

  Example code to demonstrate chained contexts. Chained operations are a
  YELLOW filled rectangled and a GREEN line.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"

#define INSTANCE 0

unsigned long context[64];      // context array

/* --------------------------------------------------------------------------
  GETPAGEPTR - calculate 32K read/write pointer from a given memory address.
-------------------------------------------------------------------------- */
unsigned long getpageptr (unsigned long addr)
{
    unsigned long pageptr;

    pageptr = addr / 0x8000;

    return (pageptr);
}

/* --------------------------------------------------------------------------
  GETOFFSET - calculate offset into 32K VGA aperture from the physical
              address and page pointer.
-------------------------------------------------------------------------- */
unsigned long getoffset (unsigned long phyaddr, unsigned long pageptr)
{
    unsigned long ptraddr, offset;

    ptraddr = pageptr * 0x8000;
    offset = phyaddr - ptraddr;

    return (offset);
}

/* --------------------------------------------------------------------------
  UPLOAD_CONTEXT - copy context array to video memory according to given
                   video memory address
-------------------------------------------------------------------------- */
void upload_context (unsigned long address)
{
    unsigned long writepage, writeoffset;
    int i;

    // Use VGA or LINEAR aperture depending on current aperture settings
    if (modeinfo.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // set write pointer to correct page
        writepage = getpageptr (address);
        writeoffset = getoffset (address, writepage);

        // upper 32k VGA aperture pointer is used for overflow
        regw (MEM_VGA_WP_SEL, ((writepage + 1) << 16) | writepage);

        // Reduce VGA aperture to 64KB addressable A000:0000-A000:FFFF as
        // some mach64CTs don't access the framebuffer properly othewise.
        outpw (0x3CE, 0x0506);      // graf06 = 05  (64K, graphics mode)

        // transfer the 256 bytes of context data to the appropriate address
        for (i = 0; i < 64; i++)
        {
            *((unsigned long far *) (LOW_APERTURE_BASE +
                                     writeoffset)) = context[i];
            writeoffset = writeoffset + 4;
        }

        // Restore VGA aperture to 128KB addressable A000:0000-B000:FFFF in
        // order to access the memory mapped registers starting at B000:FC00
        outpw (0x3CE, 0x0106);      // graf06 = 01  (128K, graphics mode)
    }
    else // Linear aperture is enabled
    {
        movemem ((void far *) (context), address + modeinfo.aperture_address,
                 256 / 2, MEM_WRITE);
    }
}


/* Main C program */

int main (int argc, char *argv[])
{
    int memcntl, i;
    unsigned long contextaddr, temp;
    unsigned long xres, yres, pitch;

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // Process the command line arguments to override default resolution
    // and color depth settings.
    process_command_line (argc, argv);

    // determine if VGA aperture or linear aperture will be used for transfer
    if (querydata.vga_type == VGA_ENABLE)
    {
        // The VGA controller is normally set in planar mode. Data transfer
        // through the VGA aperture (low and high 32K pages) requires that the
        // VGA controller be set in a packed pixel mode where the pixel data
        // is arranged contigiously.
        set_packed_pixel ();
    }

    // set an accelerator mode
    if (open_mode (gmode_res, PITCH_XRES, gclr_depth) != NO_ERROR)
    {

        // Close the mode if a packed pixel mode was set.
        if (querydata.vga_type == VGA_ENABLE)
        {
            close_mode ();
        }
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // Check for 24 bpp mode - Lines are not supported in 24 bpp modes
    if (modeinfo.bpp == 24)
    {
        // disable accelerator mode and switch back to VGA text mode
        close_mode ();

        printf ("Lines are not supported in 24 bpp modes.\n");
        return (1);
    }

    // get modal information
    xres = (unsigned long) (modeinfo.xres);
    yres = (unsigned long) (modeinfo.yres);
    pitch = (unsigned long) (modeinfo.pitch);

    // setup engine context and clear screen
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // determine top of memory address
    memcntl = ior16 (ioMEM_CNTL);
    switch (memcntl & 7)
    {
        case 0:
            contextaddr = 0x80000;      // 512K
            break;
        case 1:
            contextaddr = 0x100000;     // 1M
            break;
        case 2:
            contextaddr = 0x200000;     // 2M
            break;
        case 3:
            contextaddr = 0x400000;     // 4M
            break;
        case 4:
            contextaddr = 0x600000;     // 6M
            break;
        case 5:
            contextaddr = 0x800000;     // 8M
            break;
    }

    /*
       Fill context. Each context pointer represents 256 bytes of video
       memory. The context load address decreases as the context load pointer
       increases. Context pointer 0 points to the top of memory - 256. Some
       restrictions apply for context pointers 0-3 if the linear aperture
       size equals the memory size (i.e. 4M aperture size, 4M of video
       memory). In this case, the memory mapped registers occupy 1K of memory
       below the top of aperture. The only method to reach this area in this
       case is to use the VGA paged aperture.
    */
    context[0] = 0xFFFFFFFF;
    context[1] = 0x00000000;
    context[2] = (pitch / 8) << 22;         // destination pitch
    context[3] = 0x00200010;
    context[4] = 0x00400080;
    temp = (unsigned long) (modeinfo.xres / 20);
    context[3] = (temp << 16) | (modeinfo.yres / 30);  // (x, y)
    temp = (unsigned long) (modeinfo.xres / 10);
    context[4] = (temp << 16) | (modeinfo.yres / 4);   // (width, height)
    context[5] = 0x00000000;
    context[6] = 0x00000001;
    context[7] = 0x0003FFFF;
    context[8] = (pitch / 8) << 22;         // source pitch
    context[9] = 0x00000000;
    context[10] = 0x00000000;
    context[11] = 0x00000000;
    context[12] = 0x00000000;
    context[13] = 0x00000000;
    context[14] = 0x00000000;
    context[15] = xres << 16;               // scissors
    context[16] = yres << 16;
    context[17] = 0x00000000;
    context[18] = get_color_code(YELLOW);   // foreground color
    context[19] = 0xFFFFFFFF;

    // set DP_PIX_WIDTH and CHAIN_MASK according to color depth
    switch(modeinfo.bpp)
    {
        case 4:
            context[20] = 0x00008888;
            context[21] = HOST_4BPP | SRC_4BPP | DST_4BPP;
            break;
        case 16:
            if (modeinfo.depth == 555)
            {
                // 555 color weighting
                context[20] = 0x00004210;
                context[21] = HOST_15BPP | SRC_15BPP | DST_15BPP;
            }
            else
            {
                // 565 color weighting
                context[20] = 0x00008410;
                context[21] = HOST_16BPP | SRC_16BPP | DST_16BPP;
            }
            break;
        case 32:
            context[20] = 0x00008080;
            context[21] = HOST_32BPP | SRC_32BPP | DST_32BPP;
            break;
        case 8:
        case 24:
        default:
            context[20] = 0x00008080;
            context[21] = HOST_8BPP | SRC_8BPP | DST_8BPP;
            break;
    }

    context[22] = FRGD_MIX_S | BKGD_MIX_S;
    context[23] = FRGD_SRC_FRGD_CLR;
    context[24] = 0x00000000;
    context[25] = 0xFFFFFFFF;
    context[26] = 0x00000000;
    context[27] = DST_LAST_PEL | DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT;
    context[28] = 0x00000000;
    for (i = 29; i < 64; i++)
    {
        context[i] = 0;
    }

    // Upload contexts

    // Setup 2 contexts - one chained to the other. This is done by setting
    // up the CONTEXT_LOAD_CNTL context register (offset 28) for the current
    // context load. When a context load is initiated, the CONTEXT_LOAD_CNTL
    // register will be loaded and executed according to its contents. This
    // feature allows context chaining. In this example, the first context
    // will draw a YELLOW filled rectangle and the second context will draw
    // a LIGHTGREEN diagonal line. Since the second context is to be chained,
    // the load pointer bits of the CONTEXT_LOAD_CNTL register entry of the
    // FIRST context (load pointer = 4) will be set to point to the second
    // context (load pointer = 5).

    // Context load address calculation:
    //
    // Address = total video memory - (context pointer + 1) * 0x100)
    //

    // load up first context -> load pointer = 4
    context[28] = CONTEXT_LOAD_AND_DO_LINE | 5;   // do line for next context
    upload_context (contextaddr - ((4 + 1) * 0x100));

    // load up second context -> load pointer = 5
    temp = (unsigned long)(modeinfo.xres / 4);
    context[3] = (temp << 16) | (modeinfo.yres / 4);   // (x, y) for line
    context[18] = get_color_code (LIGHTGREEN);    // set line color
    context[28] = 0;                              // halt chain
    upload_context (contextaddr - ((5 + 1) * 0x100));

    // initiate context chain
    wait_for_fifo (2);                            // wait for 2 fifo entries
    regw (CONTEXT_MASK, 0xFFFFFFFF);
    regw (CONTEXT_LOAD_CNTL, CONTEXT_LOAD_AND_DO_FILL | 4);

    // wait for a carriage return
    getch ();

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

