/*==========================================================================
* INITMODE.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved*
*                                                                          *
* PGL functions to set and close an accelerator mode.                      *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <memory.h>
#include <dos.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_initmode - set and initialize an accelerator mode
  
  This function initializes the Mach64 hardware to the specified accelerator
  mode. The engine context is initialized according to the given input
  parameters as is an internal structure used by the engine drawing
  functions. The return value should be checked before proceeding with any
  PGL drawing functions. Functions PGL_detectmach64() and PGL_initaperture()
  MUST be called before this function.

  INPUTS:
  ======

    X resolution in pixels:

        - typical values are 640, 800, 1024, 1280, 1600
        - this value is used for screen clearing; it is NOT used for engine
          pitch or scissor sizing
        - in normal conditions, this value will be equal to the pitch

    Y resolution in lines:

        - typical values are 480, 600, 768, 1024, 1200
        - this value is used for scissor sizing and screen clearing

    Bits per pixel (bpp):

        - valid values are 4, 8, 15, 16, 24, 32
        - note that 15 gets converted to 16 with a color depth of 555

    Color depth (PGL color depth only - DAC color depth is not settable):

        - set to 0 for 4, 8, 15, 16 bpp, the value is ignored:
            4 bpp  -> set to zero internally
            8 bpp  -> set to zero internally
            15 bpp -> set to 555 internally
            16 bpp -> set to 565 internally

        - set to 0 for 24 bpp without NO_AUTO_DEPTH attribute:
            - set internally to RGB or BGR depending on DAC support

        - for 24 bpp with NO_AUTO_DEPTH attribute:
            - note that only the PGL's color depth is affected; the DAC
              color depth is set by the ROM depending on its support

                        DEPTH_24_RGB (0) - RGB
                        DEPTH_24_BGR (1) - BGR

        - set to 0 for 32 bpp without NO_AUTO_DEPTH attribute:
            - same behaviour as 24 bpp

        - for 32 bpp with NO_AUTO_DEPTH attribute:
            - note that only the PGL's color depth is affected; the DAC
              color depth is set by the ROM depending on its support

                        DEPTH_32_RGBA (0) - RGBa
                        DEPTH_32_ARGB (1) - aRGB
                        DEPTH_32_BGRA (2) - BGRa
                        DEPTH_32_ABGR (3) - aBGR

        - note that the PGL's color depth can also be set by calling
          PGL_setcolordepth() after a mode is set

    Pitch in pixels:

        - typical value equals the X resolution
        - this value is used to set the engine pitch and scissor sizing
        - this value does NOT affect the CRT pitch

    Memory offset in qwords:

        - this value is used to set the engine and CRT memory offsets
        - typically zero

    Attribute flags:

        NO_ATTRIBUTES     (0)- this is the standard value
        PALETTIZED_MODE   (1)- set palettized bit when calling ROM to set mode
        BYPASS_MODE       (2)- don't set mode but initialize graphics engine;
                               note all input parameters are used including
                               'depth'
        CLOSE_MODE        (4)- special flag to turn off accelerator mode
                             - this attribute is used by PGL_closemode()
        NO_CLEAR_SCREEN   (8)- if set, don't clear screen after setting mode
        NO_PACKED_PIXEL(1024)- don't set packed pixel mode for VGA aperture;
                               this is only for debugging purposes
        NO_AUTO_DEPTH  (2048)- use given depths for 24 or 32 bpp modes;
                               otherwise, one is selected that is supported
                               automatically

    Mode number:

        #                   - set mode using this ROM mode number
                            - the ROM sets the CRT pitch

    CRT table pointer:

        NO_CRT_TABLE (NULL) - use this input if mode number <> USE_CRT_TABLE
        pointer             - set mode with CRTC parameters from the given
                              structure if mode number = USE_CRT_TABLE

    EEPROM offset:

        NO_EEPROM_OFFSET (-1) - use this input if mode number <>
                                USE_EEPROM_OFFSET
        #                     - set mode according to this EEPROM offset index
                                value if mode number = USE_EEPROM_OFFSET


  RETURN CODES:
  ============

    MODE_SUPPORTED     - successful mode set or closure
    MODE_NOT_SUPPORTED - the selected mode is not supported
    NO_APERTURE        - an aperture is not available


  NOTES:
  =====

  - PGL_getmodeconfig() should be called after a successful call to this
    function since it will contain valuable modal information about the
    current accelerator mode (i.e. bpp and color depth)
  - the engine pitch is determined by 'pitch'
  - the engine scissor size is determined by 'yres' and 'pitch'
  - the screen clear size is determined by 'xres' and 'yres'
-------------------------------------------------------------------------- */
int PGL_initmode(int xres,
                 int yres,
                 int bpp,
                 int depth,
                 int pitch,
                 unsigned long offset,
                 int attribute_flags,
                 int mode_number,
                 PGL_crttable far *tableinfo,
                 int eeprom_offset)
{
    int retval;
    int palettized_flag;
    int bpp_code;
    int depth_support;
    int depth_error;
    int temp;
    long total_mem, mode_mem, fill_mem;
    int doubleclock_flag;

    // close accelerator mode if CLOSE_MODE flag is set (PGL_close())
    if ((attribute_flags & CLOSE_MODE) == CLOSE_MODE)
    {
        // insure DAC is set 6 bit operation
        if ((PGL_modecfg.chip_type == CHIP_CT_ID) ||
            (PGL_modecfg.chip_type == CHIP_DT_ID))
        {
            // bit must be cleared
            iow8(ioDAC_CNTL + 1, ior8(ioDAC_CNTL + 1) & 0xfe);
        }

        // ensure engine errors are cleared
        PGL_resetengine();

        // restore palette
        if (PGL_palSave == 1)
        {
            PGL_restorepalette(save_palette);

            // clear palette save flag
            PGL_palSave = 0;
        }

        // restore memory boundary
        if (PGL_modeSet == 1)
        {
            // restore hardware cursor colors
            if (PGL_savecurflag == 1)
            {
                // only restore if colors were saved
                pgl_restorecursorcolors();
            }

            // restore memory boundary setting - only touch boundary bits
            regw(MEM_CNTL, (regr(MEM_CNTL) & 0xfff8ffff) |
                           (PGL_memBndry & 0x00070000));

            // clear mode set flag
            PGL_modeSet = 0;
        }

        // turn off accelerator mode and switch to VGA, restore text mode
        pgl_preparevgaswitch();

        // set modal information in mode config structure to zero
        // *** DO NOT change aperture information ***
        PGL_modecfg.xres = 0;
        PGL_modecfg.yres = 0;
        PGL_modecfg.bpp = 0;
        PGL_modecfg.depth = 0;
        PGL_modecfg.pitch = 0;
        PGL_modecfg.mode_number = 0;
        PGL_modecfg.attribute_flags = 0;
        PGL_modecfg.actual_xres = 0;
        PGL_modecfg.actual_yres = 0;
        PGL_modecfg.maxy = 0;
        PGL_modecfg.doubleclock_flag = 0;

        return (MODE_SUPPORTED);
    }

    // assume failure
    retval = MODE_NOT_SUPPORTED;

    // get palettized flag for ROM call
    palettized_flag = attribute_flags & PALETTIZED_MODE;

    // check for aperture availability - return error if not
    if ((PGL_modecfg.linear_aperture_status == LINEAR_APERTURE_DISABLED) &&
        (PGL_modecfg.vga_aperture_status == VGA_APERTURE_DISABLED))
    {
        return (NO_APERTURE);
    }

    // check xres, yres, pitch, and mode_number for zero or negative values
    if ((xres <= 0) || (yres <= 0) || (pitch <= 0) || (mode_number <= 0))
    {
        return (retval);
    }

    // check for valid color depth values (only for 24/32 bpp modes)
    if ((attribute_flags & NO_AUTO_DEPTH) == NO_AUTO_DEPTH)
    {
        depth_error = 0;
        if (bpp == 24)
        {
            depth_error = -1;
            switch(depth)
            {
                case DEPTH_24_RGB:
                case DEPTH_24_BGR: depth_error = 0; break;
            }
        }
        if (bpp == 32)
        {
            depth_error = -1;
            switch(depth)
            {
                case DEPTH_32_RGBA:
                case DEPTH_32_ARGB:
                case DEPTH_32_BGRA:
                case DEPTH_32_ABGR: depth_error = 0; break;
            }
        }
        if (depth_error == -1)
        {
            return (retval);
        }
    }

    // check for valid bpp values and color depth support;
    bpp_code = -1;
    depth_support = -1;
    switch (bpp)
    {
        case 4:
            bpp_code = COLOR_DEPTH_4;
            depth_support = 0;
            break;

        case 8:
            bpp_code = COLOR_DEPTH_8;
            depth_support = 0;
            break;

        case 15:
            bpp_code = COLOR_DEPTH_15;
            if ((PGL_querydata.color_depth_support & COLORSUPPORT_15) == COLORSUPPORT_15)
            {
                depth_support = 0;
            }
            break;

        case 16:
            bpp_code = COLOR_DEPTH_16;
            if ((PGL_querydata.color_depth_support & COLORSUPPORT_16) == COLORSUPPORT_16)
            {
                depth_support = 0;
            }
            break;

        case 24:
            bpp_code = COLOR_DEPTH_24;
            if ((PGL_querydata.color_depth_support & COLORSUPPORT_24_RGB) == COLORSUPPORT_24_RGB)
            {
                // RGB
                depth_support = DEPTH_24_RGB;
            }
            else if ((PGL_querydata.color_depth_support & COLORSUPPORT_24_BGR) == COLORSUPPORT_24_BGR)
            {
                // BGR
                depth_support = DEPTH_24_BGR;
            }
            break;

        case 32:
            bpp_code = COLOR_DEPTH_32;
            if ((PGL_querydata.color_depth_support & COLORSUPPORT_32_RGBA) == COLORSUPPORT_32_RGBA)
            {
                // RGBa
                depth_support = DEPTH_32_RGBA;
            }
            else if ((PGL_querydata.color_depth_support & COLORSUPPORT_32_ARGB) == COLORSUPPORT_32_ARGB)
            {
                // aRGB
                depth_support = DEPTH_32_ARGB;
            }
            else if ((PGL_querydata.color_depth_support & COLORSUPPORT_32_BGRA) == COLORSUPPORT_32_BGRA)
            {
                // BGRa
                depth_support = DEPTH_32_BGRA;
            }
            else if ((PGL_querydata.color_depth_support & COLORSUPPORT_32_ABGR) == COLORSUPPORT_32_ABGR)
            {
                // aBGR
                depth_support = DEPTH_32_ABGR;
            }
            break;
    }
    if (bpp_code == -1)
    {
        return (retval);
    }
    if (depth_support == -1)
    {
        if (((attribute_flags & NO_AUTO_DEPTH) == NO_AUTO_DEPTH) ||
            ((attribute_flags & BYPASS_MODE) == BYPASS_MODE))
        {
            depth_support = depth;
        }
        else
        {
            return (retval);
        }
    }

    // 4 bpp modes are not supported on revB ATI68860 DACs
    if ((PGL_querydata.dac_type == DAC_ATI68860_1) && (bpp == 4))
    {
        return (retval);
    }

    // insure VGA aperture control bit matches aperture status variable
    // - setting a mode using INT10 may zap the VGA aperture control bit
    if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        PGL_enablevgaaperture();
    }

    // set VGA controller into packed pixel mode for data transfer
    if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // don't set packed pixel mode if flag is set
        if ((attribute_flags & NO_PACKED_PIXEL) != NO_PACKED_PIXEL)
        {
            pgl_blankvgascreen();
            pgl_setpackedpixel();
        }
    }

    // don't set accelerator mode if BYPASS_MODE flag is set
    if ((attribute_flags & BYPASS_MODE) == BYPASS_MODE)
    {
        retval = MODE_SUPPORTED;
    }
    else
    {
        // load accelerator mode parameters
        PGL_blankscreen();
        if (ROM_loadmodeparms(mode_number,
                              PITCH_XRES,
                              bpp_code,
                              palettized_flag,
                              tableinfo,
                              eeprom_offset) == NOERROR)
        {
            // set accelerator mode if supported
            ROM_setdisplaymode(ACCELERATOR_MODE,
                               palettized_flag,
                               (int far *)(&doubleclock_flag));

            // set doubleclock_flag
            PGL_modecfg.doubleclock_flag = doubleclock_flag;

            retval = MODE_SUPPORTED;
        }
        PGL_unblankscreen();
    }

    // setup engine if setmode was successful or if bypass flag is set
    if (retval == MODE_SUPPORTED)
    {
        // set mode setting flag
        PGL_modeSet = 1;

        // set shared memory boundary so all memory can be used for
        // accelerator; touch only the boundary bits
        PGL_memBndry = regr(MEM_CNTL);
        regw(MEM_CNTL, PGL_memBndry & 0xfff8ffff);

        // initialize PGL_modecfg structure with modal information
        PGL_modecfg.mode_number = mode_number;
        PGL_modecfg.xres = xres;
        PGL_modecfg.yres = yres;
        switch(bpp)
        {
            case 4:
            case 8:
                PGL_modecfg.bpp = bpp;
                PGL_modecfg.depth = 0;
                break;

            case 15:
                PGL_modecfg.bpp = 16;
                PGL_modecfg.depth = 555;
                break;

            case 16:
                PGL_modecfg.bpp = bpp;
                PGL_modecfg.depth = 565;
                break;

            case 24:
            case 32:
                PGL_modecfg.bpp = bpp;
                PGL_modecfg.depth = depth_support;
                break;
        }
        PGL_modecfg.pitch = pitch;
        PGL_modecfg.memory_offset = offset;
        PGL_modecfg.attribute_flags = attribute_flags;

        // set 'actual size' variables
        PGL_modecfg.actual_xres = (int)(((regr(CRTC_H_TOTAL_DISP) >> 16) + 1) * 8);
        PGL_modecfg.actual_yres = (int)((regr(CRTC_V_TOTAL_DISP) >> 16) + 1);

        // Some modes may be double-clocked meaning that the actual x
        // resolution is double that is H_DISP. This is a patch to fix
        // this returned value. 'pitch' may be set to a larger value to
        // permit a larger desktop.
        //
        if (doubleclock_flag == DOUBLE_CLOCK)
        {
            // actual xres is twice as large
            PGL_modecfg.actual_xres = PGL_modecfg.actual_xres * 2;
        }
        else
        {
            // this is the backup patch if the ROM does not support the
            // double clock return flag
            //
            if (PGL_modecfg.actual_yres > 1000)
            {
                if (PGL_modecfg.actual_xres < PGL_modecfg.actual_yres)
                {
                    // actual xres is twice as large
                    PGL_modecfg.actual_xres = PGL_modecfg.actual_xres * 2;
                }
            }
        }

        // ensure that vga page pointers are set to page zero and sized to 64K
        regw(MEM_VGA_WP_SEL, 0x00010000);
        regw(MEM_VGA_RP_SEL, 0x00010000);

        // initialize engine context
        PGL_initengine();

        // clear screen if no clear screen flag is not set
        if ((attribute_flags & NO_CLEAR_SCREEN) != NO_CLEAR_SCREEN)
        {
            PGL_clearscreen(0, 0, xres, yres);
        }

        // initialize palette depending on hardware and flags set
        if ((PGL_modecfg.bpp == 4) || (PGL_modecfg.bpp == 8))
        {
            // set 6 bit DAC operation for 4 and 8 bpp modes
            if ((PGL_modecfg.chip_type == CHIP_CT_ID) ||
                (PGL_modecfg.chip_type == CHIP_DT_ID))
            {
                // bit must be cleared
                iow8(ioDAC_CNTL + 1, ior8(ioDAC_CNTL + 1) & 0xfe);
            }

            // save the current palette
            PGL_savepalette(save_palette);
            PGL_palSave = 1;

            // initialize palette
            PGL_initpalette();
        }
        else  // hi color modes (15, 16, 24, 32 bpp)
        {
            // set palette if palettized flag is set depending palette flags
            if (((attribute_flags & PALETTIZED_MODE) == PALETTIZED_MODE) ||
                (PGL_modecfg.chip_type == CHIP_CT_ID) ||
                (PGL_modecfg.chip_type == CHIP_DT_ID))
            {
                // set 8 bit DAC operation for palettized 15, 16, 24, 32 modes
                if ((PGL_modecfg.chip_type == CHIP_CT_ID) ||
                    (PGL_modecfg.chip_type == CHIP_DT_ID))
                {
                    // bit must be set
                    iow8(ioDAC_CNTL + 1, ior8(ioDAC_CNTL + 1) | 0x01);
                }

                // save the current palette
                PGL_savepalette(save_palette);
                PGL_palSave = 1;

                // initialize palette
                PGL_initpalettized();
            }
        }

        /* determine maximum y line (top of video memory) */

        // get total video memory size in bytes
        temp = (int)(regr(MEM_CNTL) & 0x07);

        // Mach64 CT, DT chips can support upto 4Meg only
        if ((PGL_modecfg.chip_type == CHIP_CT_ID) ||
            (PGL_modecfg.chip_type == CHIP_DT_ID))
        {
            temp = temp & 0x03;
        }

        // assign total video memory size on card
        switch(temp)
        {
            case 0: total_mem = 1L << 19; break;           // 512K
            case 1: total_mem = 1L << 20; break;           // 1M
            default:
            case 2: total_mem = 1L << 21; break;           // 2M
            case 3: total_mem = 1L << 22; break;           // 4M
            case 4: total_mem = (1L << 22) + (1L << 21);   // 6M
            case 5: total_mem = 1L << 23; break;           // 8M
        }

        // get memory size used by selected mode in bytes
        mode_mem = (long)(yres);
        if (PGL_modecfg.bpp == 4)
        {
            // 4 bpp
            mode_mem = (long)((mode_mem * pitch)/2);
        }
        else
        {
            // 8, 15, 16, 24, 32 bpp
            mode_mem = (long)(mode_mem * pitch * (PGL_modecfg.bpp / 8));
        }

        // add 'memory offset' (in qwords) to mode memory (in bytes)
        mode_mem = mode_mem + (PGL_modecfg.memory_offset * 8);

        // get off-screen accelerator memory available in bytes
        if (mode_mem > total_mem)
        {
            fill_mem = 0;
        }
        else
        {
            fill_mem = total_mem - mode_mem;
        }

        // convert off-screen accelerator memory into modal lines
        if (PGL_modecfg.bpp == 4)
        {
            fill_mem = (long)((fill_mem * 2)/pitch);
        }
        else
        {
            fill_mem = (long)(fill_mem/(pitch * (PGL_modecfg.bpp / 8)));
        }

        // calculate maximum line that can be used for off-screen memory
        temp = (int)(fill_mem);
        PGL_modecfg.maxy = yres;
        if (temp > 0)
        {
            PGL_modecfg.maxy = PGL_modecfg.maxy + temp - 1;
            if (PGL_modecfg.maxy > ENGINE_MAX_Y)
            {
                PGL_modecfg.maxy = ENGINE_MAX_Y;
            }
        }

        // other initializations
        PGL_setfgcolor(PGL_getcolorcode(WHITE));
        PGL_setbgcolor(BLACK);

        // save hardware cursor colors
        if (PGL_savecurflag == 0)
        {
            // only save for first instance of accelerator mode set
            PGL_savecurflag = 1;
            pgl_savecursorcolors();
        }
    }

    // restore text mode if mode set failed
    if (retval == MODE_NOT_SUPPORTED)
    {
        pgl_preparevgaswitch();
    }

    // return status code
    return (retval);
}

/* --------------------------------------------------------------------------
  PGL_closemode - close the current accelerator mode
  
  This function turns off the current accelerator mode & switches the
  display back to VGA.
-------------------------------------------------------------------------- */
void PGL_closemode(void)
{
    PGL_initmode(0, 0, 0, 0, 0, 0, CLOSE_MODE, 0, NO_CRT_TABLE, NO_EEPROM_OFFSET);
}

/* --------------------------------------------------------------------------
  pgl_preparevgaswitch - setup for accelerator mode exit for the PGL
-------------------------------------------------------------------------- */
void pgl_preparevgaswitch(void)
{
    union REGS regs;
    int vga_status;
    int doubleclock_flag;

    // VGA aperture must be disabled when calling ROM to switch to VGA
    vga_status = PGL_modecfg.vga_aperture_status;
    if (vga_status == VGA_APERTURE_ENABLED)
    {
        PGL_disablevgaaperture();
    }

    // Disable accelerator mode, switch back to VGA mode -- ignore the
    // double clock flag in VGA mode
    //
    ROM_setdisplaymode(VGA_MODE,
                       PALETTIZE_DISABLE,
                       (int far *)(&doubleclock_flag));

    // ensure screen is unblanked
    pgl_unblankvgascreen();

    if (PGL_modeType == 7)
    {
        // clear color text mode
        regs.x.ax = 3;
        int86(0x10, &regs, &regs);
    }

    // restore text mode saved previously
    pgl_restoretextmode();

    // restore VGA aperture state
    if (vga_status == VGA_APERTURE_ENABLED)
    {
        PGL_enablevgaaperture();
    }
}

